/* COPYRIGHT 2000 by 3rd-evolution
 * Author: Thomas Krammer (tkrammer@3rd-evolution.de)
 */

#include "SerialDisplay.h"

const char * const COMMAND_AUTO_WRAP_ON					=	"\xFE\x43";
const char * const COMMAND_AUTO_WRAP_OFF				=	"\xFE\x44";
const char * const COMMAND_AUTO_SCROLL_ON				=	"\xFE\x51";
const char * const COMMAND_AUTO_SCROLL_OFF				=	"\xFE\x52";
const char * const COMMAND_BACKLIGHT_ON					=	"\xFE\x42";
const char * const COMMAND_BACKLIGHT_OFF				=	"\xFE\x46";
const char * const COMMAND_CLEAR_DISPLAY				=	"\xFE\x58";
const char * const COMMAND_CURSOR_ON					=	"\xFE\x4A";
const char * const COMMAND_CURSOR_OFF					=	"\xFE\x4B";
const char * const COMMAND_CURSOR_BLINK_ON				=	"\xFE\x53";
const char * const COMMAND_CURSOR_BLINK_OFF				=	"\xFE\x54";
const char * const COMMAND_CURSOR_LEFT					=	"\xFE\x4C";
const char * const COMMAND_CURSOR_RIGHT					=	"\xFE\x4D";
const char * const COMMAND_CURSOR_TO					=	"\xFE\x47";
const char * const COMMAND_HOME							=	"\xFE\x48";
const char * const COMMAND_INIT_HORIZONTAL_BAR_GRAPH	= 	"\xFE\x68";
const char * const COMMAND_INIT_VERTICAL_BAR_GRAPH		=	"\xFE\x76";
const char * const COMMAND_INIT_THIN_VERT_BAR_GRAPH		=	"\xFE\x73";
const char * const COMMAND_DRAW_HORIZONTAL_BAR			=	"\xFE\x7C";
const char * const COMMAND_DRAW_VERTICAL_BAR			=	"\xFE\x3D";

// === global ===

// This small helper function converts a data_rate enum into bps.
int32 data_rate_in_bps(data_rate dataRate)
{
	switch(dataRate) {
		case B_0_BPS:
			return 0;
		case B_50_BPS:
			return 50;
		case B_75_BPS:
			return 75;
		case B_110_BPS:
			return 110;
		case B_134_BPS:
			return 134;
		case B_150_BPS:
			return 150;
		case B_200_BPS:
			return 200;
		case B_300_BPS:
			return 300;
		case B_600_BPS:
			return 600;
		case B_1200_BPS:
			return 1200;
		case B_1800_BPS:
			return 1800;
		case B_2400_BPS:
			return 2400;
		case B_4800_BPS:
			return 4800;
		case B_9600_BPS:
			return 9600;
		case B_19200_BPS:
			return 19200;
		case B_38400_BPS:
			return 38400;
		case B_57600_BPS:
			return 57600;
		case B_115200_BPS:
			return 115200;
		case B_230400_BPS:
			return 230400;
		case B_31250_BPS:
			return 31250;
	}
	
	return 0;
}

// ==== CMatrixOrbitalSerialDisplay ====

CMatrixOrbitalSerialDisplay::CMatrixOrbitalSerialDisplay()
{
	serial = NULL;
	
	initResult = B_NO_INIT;
}

CMatrixOrbitalSerialDisplay::~CMatrixOrbitalSerialDisplay()
{
	delete serial;
}

status_t CMatrixOrbitalSerialDisplay::Init(int32 port)
{
	char deviceName[B_OS_NAME_LENGTH];

	serial = new BSerialPort();

	serial->GetDeviceName(port, deviceName);
	
	if((initResult = serial->Open(deviceName)) > 0) {
		// COM port successfully opened.
		initResult = CommonInit();
	}
	
	return initResult;
}

status_t CMatrixOrbitalSerialDisplay::Init(const char *deviceName)
{
	serial = new BSerialPort();

	if((initResult = serial->Open(deviceName)) > 0) {
		// COM port successfully opened.
		initResult = CommonInit();
	}

	return initResult;
}

status_t CMatrixOrbitalSerialDisplay::CommonInit()
{
	serial->SetFlowControl(B_HARDWARE_CONTROL);
	serial->SetDataBits(B_DATA_BITS_8);		// 8N1
	serial->SetParityMode(B_NO_PARITY);
	serial->SetStopBits(B_STOP_BITS_1);
	serial->SetDataRate(B_19200_BPS);		// 19200 bps
	
	return B_OK;
}

void CMatrixOrbitalSerialDisplay::AutoWrap(bool enable)
{
	Write(enable ? COMMAND_AUTO_WRAP_ON : COMMAND_AUTO_WRAP_OFF, 2);
}

void CMatrixOrbitalSerialDisplay::AutoScroll(bool enable)
{
	Write(enable ? COMMAND_AUTO_SCROLL_ON : COMMAND_AUTO_SCROLL_OFF, 2);
}

void CMatrixOrbitalSerialDisplay::Backlight(bool enable)
{
	if(enable) {
		char buffer[3];

		strcpy(buffer, COMMAND_BACKLIGHT_ON);		
		buffer[2] = '\0';	// no powersave.
		
		Write(buffer, 3);
	} else {
		Write(COMMAND_BACKLIGHT_OFF, 2);
	}
}

void CMatrixOrbitalSerialDisplay::Clear()
{
	Write(COMMAND_CLEAR_DISPLAY, 2);
}

void CMatrixOrbitalSerialDisplay::Home()
{
	Write(COMMAND_HOME, 2);
}

void CMatrixOrbitalSerialDisplay::Cursor(bool enable, bool blink)
{
	Write(enable ? COMMAND_CURSOR_ON : COMMAND_CURSOR_OFF, 2);
	Write(blink ? COMMAND_CURSOR_BLINK_ON : COMMAND_CURSOR_BLINK_OFF, 2);
}

void CMatrixOrbitalSerialDisplay::Write(const char *cmd, int32 numBytes)
{
	if(serial)
		serial->Write((const void *)cmd, numBytes);
}

void CMatrixOrbitalSerialDisplay::DisplayString(const char *str)
{
	Write(str, strlen(str));
}

void CMatrixOrbitalSerialDisplay::DisplayString(const char *str, int8 x, int8 y)
{
	CursorTo(x,y);
	Write(str, strlen(str));
}

void CMatrixOrbitalSerialDisplay::DisplayString(const char *str, int8 x, int8 y, int8 len)
{
	CursorTo(x,y);
	Write(str, MIN(len, (int8)strlen(str)));
}

void CMatrixOrbitalSerialDisplay::CursorTo(int8 x, int8 y)
{
	char command[4];
	
	command[0] = COMMAND_CURSOR_TO[0];	
	command[1] = COMMAND_CURSOR_TO[1];	
	command[2] = x+1;
	command[3] = y+1;
	
	Write(command, 4);
}

void CMatrixOrbitalSerialDisplay::InitHorizontalBarGraph()
{
	Write(COMMAND_INIT_HORIZONTAL_BAR_GRAPH, 2);
}

void CMatrixOrbitalSerialDisplay::InitVerticalBarGraph()
{
	Write(COMMAND_INIT_VERTICAL_BAR_GRAPH, 2);
}

void CMatrixOrbitalSerialDisplay::InitThinVerticalBarGraph()
{
	Write(COMMAND_INIT_THIN_VERT_BAR_GRAPH, 2);
}
	
void CMatrixOrbitalSerialDisplay::DrawVerticalBar(int8 column, int8 length)
{
	char command[4];
	
	command[0] = COMMAND_DRAW_VERTICAL_BAR[0];
	command[1] = COMMAND_DRAW_VERTICAL_BAR[1];
	command[2] = column+1;
	command[3] = length;

	Write(command, 4);	
}

void CMatrixOrbitalSerialDisplay::DrawHorizontalBar(int8 column, int8 row, bool rightToLeft, int8 length)
{
	char command[6];
	
	command[0] = COMMAND_DRAW_HORIZONTAL_BAR[0];
	command[1] = COMMAND_DRAW_HORIZONTAL_BAR[1];
	command[2] = column+1;
	command[3] = row+1;
	command[4] = rightToLeft ? 1 : 0;
	command[5] = length;

	Write(command, 6);	
}

